/**
* Script: camera.js
* Written by: Radnen
* Updated: 10/26/2010
**/

/**
* Custom Camera Control
*  - Used in lieu of AttachCamera(name);
**/
var Camera = ({
	x: 0,
	xd: 0,
	nx: 0,
	xoff: 0,
	y: 0,
	yd: 0,
	ny: 0,
	yoff: 0,
	time: 0,
	input: null,
	msecs: 0,
	done: true,

	/**
	* attach(name);
	*  - attaches camera to person named 'name'.
	**/
	attach: function(name) {
		this.input = name;
		this.x = GetPersonX(name);
		this.xd = this.x;
		this.y = GetPersonY(name);
		this.yd = this.y;
		if (IsMapEngineRunning()) this.set(this.x, this.y);
	},
	
	/**
	* detatch();
	*  - releases camera from attached entity.
	**/
	detach: function() {
		this.input = null;
	},
	
	/**
	* pan(x, y, msecs);
	*  - pans the camera to location ('x', 'y') in 'msecs' milliseconds.
	**/
	pan: function(x, y, msecs) {
		this.input = null;
		this.msecs = msecs;
		this.nx = x; this.ny = y;
		this.time = GetTime();
		this.done = false;
	},
	
	/**
	* panToTile(x, y, msecs);
	*  - same as pan, but with regard to tile coords.
	**/
	panToTile: function(x, y, msecs) {
		this.input = null;
		this.msecs = msecs;
		this.nx = x*GetTileWidth();
		this.ny = y*GetTileHeight();
		this.time = GetTime();
		this.done = false;
	},
	
	/**
	* update();
	*  - Goes in an update script.
	**/
	update: function() {
		if (this.input != null && DoesPersonExist(this.input)) {
			this.done = true;
			var px = GetPersonX(this.input);
			var py = GetPersonY(this.input);
			this.set(px+this.xoff, py+this.yoff);
			this.y = GetCameraY(); this.x = GetCameraX();
			this.xd = this.x; this.yd = this.y;
		}
		else if (this.time + this.msecs > GetTime()) {
			with(this) {
				if (nx > x) xd = x + (GetTime() - time) * (nx-x) / msecs;
				if (nx < x) xd = x - (GetTime() - time) * (x-nx) / msecs;
				if (ny > y) yd = y + (GetTime() - time) * (ny-y) / msecs;
				if (ny < y) yd = y - (GetTime() - time) * (y-ny) / msecs;
			}
			this.set(this.xd+this.xoff, this.yd+this.yoff);
		}
		else {
			this.done = true;
			this.set(this.nx+this.xoff, this.ny+this.yoff);
			this.y = GetCameraY(); this.x = GetCameraX();
		}
	},
	
	/**
	* panToPerson(name, msecs [, wait]);
	*  - pans the camera to entity 'name's position in 'msecs' milliseconds,
	*		 optionally set to wait or not. 
	**/
	panToPerson: function(name, msecs, wait) {
		if (wait === undefined) wait = true;
		var px = GetPersonX(name);
		var py = GetPersonY(name);
		this.pan(px, py, msecs);
		if (wait) this.waitForMe();
	},
	
	/* internal : do not use */
	set: function(x, y) {
		if (x != Number.POSITIVE_INFINITY || y != Number.POSITIVE_INFINITY) {
			SetCameraX(x);
			SetCameraY(y);
		}
	},
	
	/**
	* setXY(x, y);
	*  - sets the cameras x and y pixel position.
	**/
	setXY: function(x, y) {
		this.set(x, y);
		this.x = x;
		this.y = y;
		this.nx = x;
		this.ny = y;
		this.xd = x;
		this.yd = y;
	},
	
	/**
	* waitForMe();
	*  - waits for the camera to finish by only updating the map engine while its moving.
	**/
	waitForMe: function() {
		var LastFr = GetFrameRate();
		SetFrameRate(GetMapEngineFrameRate());
		while(!this.done) {
			UpdateMapEngine();
			RenderMap();
			this.update();
			FlipScreen();
		}
		SetFrameRate(LastFr);
	},
	
	/* returns x coord */
	getX: function() {
		return this.x + this.xoff;
	},
	
	/* returns y coord */
	getY: function() {
		return this.y + this.yoff;
	}
});